<?php
/**
 * Settings DAO.
 *
 * Provides tenant-scoped reads/writes for the nxtcc_user_settings table.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

/**
 * Data access with caching for nxtcc_user_settings.
 *
 * Treats user_mailid as the unique tenant key:
 * - One logical row per user.
 * - Edits always update the latest row for that user instead of inserting.
 */
final class NXTCC_Settings_DAO {

	/**
	 * Table name (prefixed).
	 *
	 * @var string
	 */
	private static string $table = '';

	/**
	 * Initialise table name.
	 *
	 * @return void
	 */
	public static function boot(): void {
		self::$table = NXTCC_DB_AdminSettings::prefix() . 'nxtcc_user_settings';
	}

	/**
	 * Quote a table identifier for SQL usage.
	 *
	 * @param string $table Table name.
	 * @return string Backtick-quoted table name.
	 */
	private static function quote_table_name( string $table ): string {
		$clean = preg_replace( '/[^A-Za-z0-9_]/', '', $table );
		if ( ! is_string( $clean ) || '' === $clean ) {
			$clean = 'nxtcc_invalid';
		}

		return '`' . $clean . '`';
	}

	/**
	 * Prepare SQL that contains a single table placeholder token.
	 *
	 * @param string $query     SQL containing a %s table placeholder.
	 * @param string $table_sql Backtick-quoted table name.
	 * @param mixed  ...$args   Additional placeholders.
	 * @return string Prepared SQL, or empty string on failure.
	 */
	private static function prepare_with_table_token( string $query, string $table_sql, ...$args ): string {
		global $wpdb;

		$prepare_args = array_merge( array( '__NXTCC_TABLE__' ), $args );
		$prepared     = call_user_func_array(
			array( $wpdb, 'prepare' ),
			array_merge( array( $query ), $prepare_args )
		);

		if ( ! is_string( $prepared ) || '' === $prepared ) {
			return '';
		}

		return str_replace(
			array( "'__NXTCC_TABLE__'", '__NXTCC_TABLE__' ),
			$table_sql,
			$prepared
		);
	}

	/**
	 * Check whether a column exists on the settings table.
	 *
	 * @param string $column Column name.
	 * @return bool
	 */
	private static function has_column( string $column ): bool {
		global $wpdb;

		$table = self::quote_table_name( self::$table );
		$query = self::prepare_with_table_token(
			'SHOW COLUMNS FROM %s LIKE %s',
			$table,
			$column
		);

		if ( '' === $query ) {
			return false;
		}

		$val = call_user_func( array( $wpdb, 'get_var' ), $query );

		return is_string( $val ) && '' !== $val;
	}

	/**
	 * Whether encrypted app-secret columns are available.
	 *
	 * @return bool
	 */
	public static function supports_app_secret_columns(): bool {
		static $supported = null;

		if ( null !== $supported ) {
			return (bool) $supported;
		}

		$supported = self::has_column( 'app_secret_ct' ) && self::has_column( 'app_secret_nonce' );

		return (bool) $supported;
	}

	/**
	 * Check whether a tenant already has a stored encrypted App Secret.
	 *
	 * @param string $user_mailid User email.
	 * @param string $baid        Business account ID.
	 * @param string $pnid        Phone number ID.
	 * @return bool
	 */
	public static function has_saved_app_secret_for_tenant( string $user_mailid, string $baid, string $pnid ): bool {
		global $wpdb;

		if ( '' === $user_mailid || '' === $baid || '' === $pnid ) {
			return false;
		}

		if ( ! self::supports_app_secret_columns() ) {
			return false;
		}

		$table = self::quote_table_name( self::$table );
		$query = self::prepare_with_table_token(
			'SELECT COUNT(*)
			   FROM %s
			  WHERE user_mailid = %s
			    AND business_account_id = %s
			    AND phone_number_id = %s
			    AND app_secret_ct <> %s
			    AND app_secret_nonce IS NOT NULL',
			$table,
			$user_mailid,
			$baid,
			$pnid,
			''
		);

		if ( '' === $query ) {
			return false;
		}

		$count = call_user_func( array( $wpdb, 'get_var' ), $query );

		return ( (int) $count ) > 0;
	}

	/**
	 * Cache key for "latest by user".
	 *
	 * @param string $user_mailid User email.
	 * @return string
	 */
	private static function ck_latest_by_user( string $user_mailid ): string {
		return 'latest_by_user:' . md5( $user_mailid );
	}

	/**
	 * Get latest settings row for a user (cached).
	 *
	 * @param string $user_mailid User email.
	 * @return object|null
	 */
	public static function get_latest_for_user( string $user_mailid ) {
		$ck = self::ck_latest_by_user( $user_mailid );

		$table = self::quote_table_name( self::$table );
		$query = self::prepare_with_table_token(
			'SELECT * FROM %s WHERE user_mailid = %s ORDER BY id DESC LIMIT 1',
			$table,
			$user_mailid
		);
		if ( '' === $query ) {
			return null;
		}

		// TTL is a literal 300+ inside the wrapper to satisfy VIP sniffs.
		return NXTCC_DB_AdminSettings::get_row_prepared_query( $query, $ck, 300 );
	}

	/**
	 * Insert or update verify token hash for a given user (tenant).
	 *
	 * @param string $user_mailid User email.
	 * @param string $baid        Business account ID.
	 * @param string $pnid        Phone number ID.
	 * @param string $verify_hash SHA-256 hash of verify token.
	 * @return bool
	 */
	public static function upsert_verify_token_hash( string $user_mailid, string $baid, string $pnid, string $verify_hash ): bool {
		if ( '' === $user_mailid ) {
			return false;
		}

		$now_utc  = current_time( 'mysql', 1 );
		$existing = self::get_latest_for_user( $user_mailid );

		$data = array(
			'user_mailid'                    => $user_mailid,
			'business_account_id'            => $baid,
			'phone_number_id'                => $pnid,
			'meta_webhook_verify_token_hash' => $verify_hash,
			'meta_webhook_subscribed'        => 1,
			'updated_at'                     => $now_utc,
		);

		$ok = false;

		if ( $existing && ! empty( $existing->id ) ) {
			$ok = ( false !== NXTCC_DB_AdminSettings::update(
				self::$table,
				$data,
				array(
					'id' => (int) $existing->id,
				)
			) );
		} else {
			$data['created_at'] = $now_utc;
			$ok                 = ( false !== NXTCC_DB_AdminSettings::insert( self::$table, $data ) );
		}

		if ( $ok ) {
			NXTCC_DB_AdminSettings::cache_delete( self::ck_latest_by_user( $user_mailid ) );
		}

		return (bool) $ok;
	}

	/**
	 * Insert or update generic settings row for this user.
	 *
	 * @param array $data Data for insert/update.
	 * @return bool
	 */
	public static function upsert_settings( array $data ): bool {
		$user_mailid = (string) ( $data['user_mailid'] ?? '' );

		if ( '' === $user_mailid ) {
			return false;
		}

		$now_utc  = current_time( 'mysql', 1 );
		$existing = self::get_latest_for_user( $user_mailid );

		$data['updated_at'] = $now_utc;
		$ok                 = false;

		if ( $existing && ! empty( $existing->id ) ) {
			$ok = ( false !== NXTCC_DB_AdminSettings::update(
				self::$table,
				$data,
				array(
					'id' => (int) $existing->id,
				)
			) );
		} else {
			$data['created_at'] = $now_utc;
			$ok                 = ( false !== NXTCC_DB_AdminSettings::insert( self::$table, $data ) );
		}

		if ( $ok ) {
			NXTCC_DB_AdminSettings::cache_delete( self::ck_latest_by_user( $user_mailid ) );
		}

		return (bool) $ok;
	}
}

NXTCC_Settings_DAO::boot();
