<?php
/**
 * DAO helpers used by send message functions.
 *
 * This file is OO-only to satisfy PHPCS rules.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

require_once NXTCC_PLUGIN_DIR . 'includes/class-nxtcc-db-sendmessage.php';

/**
 * DAO helpers used by send message functions.
 */
final class NXTCC_Send_DAO {

	/**
	 * Quote a table identifier for SQL usage.
	 *
	 * @param string $table Table name.
	 * @return string Backtick-quoted table name.
	 */
	private static function quote_table_name( string $table ): string {
		$clean = preg_replace( '/[^A-Za-z0-9_]/', '', $table );
		if ( ! is_string( $clean ) || '' === $clean ) {
			$clean = 'nxtcc_invalid';
		}

		return '`' . $clean . '`';
	}

	/**
	 * Prepare SQL that contains a single table placeholder token.
	 *
	 * @param string $query     SQL containing a %s table placeholder.
	 * @param string $table_sql Backtick-quoted table name.
	 * @param mixed  ...$args   Additional placeholders.
	 * @return string Prepared SQL, or empty string on failure.
	 */
	private static function prepare_with_table_token( string $query, string $table_sql, ...$args ): string {
		global $wpdb;

		$prepare_args = array_merge( array( '__NXTCC_TABLE__' ), $args );
		$prepared     = call_user_func_array(
			array( $wpdb, 'prepare' ),
			array_merge( array( $query ), $prepare_args )
		);

		if ( ! is_string( $prepared ) || '' === $prepared ) {
			return '';
		}

		return str_replace(
			array( "'__NXTCC_TABLE__'", '__NXTCC_TABLE__' ),
			$table_sql,
			$prepared
		);
	}

	/**
	 * Fetch the latest encrypted token row for a tenant.
	 *
	 * @param string $user_mailid         User email.
	 * @param string $business_account_id Business account ID.
	 * @param string $phone_number_id     Phone number ID.
	 * @return object|null
	 */
	public static function get_settings_row( string $user_mailid, string $business_account_id, string $phone_number_id ): ?object {
		$ck = 'settings_row:' . md5( $user_mailid . '|' . $business_account_id . '|' . $phone_number_id );

		global $wpdb;
		$table = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );

		$prepared = self::prepare_with_table_token(
			'SELECT access_token_ct, access_token_nonce
						   FROM %s
						  WHERE user_mailid = %s AND business_account_id = %s AND phone_number_id = %s
					   ORDER BY id DESC
						  LIMIT 1',
			$table,
			$user_mailid,
			$business_account_id,
			$phone_number_id
		);
		if ( '' === $prepared ) {
			return null;
		}

		return NXTCC_DB_SendMessage::get_row_prepared_sql( $prepared, $ck );
	}

	/**
	 * Fetch contact phone fields for recipient formatting.
	 *
	 * @param int    $contact_id  Contact ID.
	 * @param string $user_mailid User email.
	 * @return object|null
	 */
	public static function get_contact_row( int $contact_id, string $user_mailid ): ?object {
		$ck = 'contact_row:' . md5( (string) $contact_id . '|' . $user_mailid );

		global $wpdb;
		$table = self::quote_table_name( $wpdb->prefix . 'nxtcc_contacts' );

		$prepared = self::prepare_with_table_token(
			'SELECT phone_number, country_code
						   FROM %s
						  WHERE id = %d AND user_mailid = %s',
			$table,
			$contact_id,
			$user_mailid
		);
		if ( '' === $prepared ) {
			return null;
		}

		return NXTCC_DB_SendMessage::get_row_prepared_sql( $prepared, $ck );
	}

	/**
	 * Resolve local history ID from Meta WAMID.
	 *
	 * @param string $wamid Meta message ID.
	 * @return int
	 */
	public static function get_history_id_by_wamid( string $wamid ): int {
		$ck = 'mh_id_by_wamid:' . md5( $wamid );

		global $wpdb;
		$table = self::quote_table_name( $wpdb->prefix . 'nxtcc_message_history' );

		$prepared = self::prepare_with_table_token(
			'SELECT id
						   FROM %s
						  WHERE meta_message_id = %s
						  LIMIT 1',
			$table,
			$wamid
		);
		if ( '' === $prepared ) {
			return 0;
		}

		return (int) NXTCC_DB_SendMessage::get_var_prepared_sql( $prepared, $ck );
	}

	/**
	 * Insert a message-history row.
	 *
	 * @param array $row Insert row.
	 * @return int|false
	 */
	public static function insert_history( array $row ) {
		global $wpdb;

		$table = $wpdb->prefix . 'nxtcc_message_history';
		return NXTCC_DB_SendMessage::insert( $table, $row );
	}
}
