<?php
/**
 * Authentication DAO.
 *
 * Encapsulates all database access for authentication/OTP/bindings/history.
 * Keeps $wpdb usage centralized and callable via prepared queries.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class NXTCC_Auth_DAO
 *
 * Database access layer for auth-related tables.
 */
final class NXTCC_Auth_DAO {

	/**
	 * Return a single row as an associative array.
	 *
	 * @param string $prepared Prepared SQL string.
	 * @return array|null Row array or null.
	 */
	private static function row( string $prepared ): ?array {
		global $wpdb;

		$row = call_user_func( array( $wpdb, 'get_row' ), $prepared, ARRAY_A );

		return is_array( $row ) ? $row : null;
	}

	/**
	 * Return many rows as associative arrays.
	 *
	 * @param string $prepared Prepared SQL string.
	 * @return array<int, array> Rows.
	 */
	private static function results( string $prepared ): array {
		global $wpdb;

		$rows = call_user_func( array( $wpdb, 'get_results' ), $prepared, ARRAY_A );

		return is_array( $rows ) ? $rows : array();
	}

	/**
	 * Return a scalar value.
	 *
	 * @param string $prepared Prepared SQL string.
	 * @return mixed Scalar value.
	 */
	private static function var_value( string $prepared ) {
		global $wpdb;

		return call_user_func( array( $wpdb, 'get_var' ), $prepared );
	}

	/**
	 * Quote a table identifier for SQL usage.
	 *
	 * @param string $table Table name.
	 * @return string Backtick-quoted table name.
	 */
	private static function quote_table_name( string $table ): string {
		$clean = preg_replace( '/[^A-Za-z0-9_]/', '', $table );
		if ( ! is_string( $clean ) || '' === $clean ) {
			$clean = $table;
		}
		return '`' . $clean . '`';
	}

	/**
	 * Prepare SQL that uses a table placeholder token.
	 *
	 * We pass a sentinel token through wpdb::prepare(), then replace quoted or
	 * unquoted sentinel forms with a pre-quoted table identifier.
	 * Use `{table}` in queries for table identifiers; this method expands each
	 * marker to a prepared `%s` and injects the sentinel for every marker.
	 *
	 * @param string $query     SQL containing table placeholder(s).
	 * @param string $table_sql Quoted table identifier.
	 * @param mixed  ...$args   Additional prepare args (for non-table placeholders).
	 * @return string Prepared SQL.
	 */
	private static function prepare_with_table_token( string $query, string $table_sql, ...$args ): string {
		global $wpdb;

		$table_marker = '{table}';
		$table_count  = substr_count( $query, $table_marker );
		$table_args   = array();

		if ( 0 < $table_count ) {
			$query      = str_replace( $table_marker, '%s', $query );
			$table_args = array_fill( 0, $table_count, '__NXTCC_TABLE__' );
		} else {
			/*
			 * Backward-compatibility fallback: if no marker is present, treat the
			 * first placeholder as the table placeholder.
			 */
			$table_args = array( '__NXTCC_TABLE__' );
		}

		$prepare_args = array_merge( $table_args, $args );
		$prepared     = call_user_func_array(
			array( $wpdb, 'prepare' ),
			array_merge( array( $query ), $prepare_args )
		);

		if ( ! is_string( $prepared ) || '' === $prepared ) {
			return '';
		}

		return str_replace(
			array( "'__NXTCC_TABLE__'", '__NXTCC_TABLE__' ),
			$table_sql,
			$prepared
		);
	}

	/**
	 * Insert row.
	 *
	 * @param string             $table  Table name (with prefix).
	 * @param array              $data   Row data.
	 * @param array<int, string> $format Format array.
	 * @return int|false Insert result.
	 */
	private static function insert( string $table, array $data, array $format ) {
		global $wpdb;

		return call_user_func( array( $wpdb, 'insert' ), $table, $data, $format );
	}

	/**
	 * Update rows.
	 *
	 * @param string                  $table        Table name (with prefix).
	 * @param array                   $data         Data map.
	 * @param array                   $where        Where map.
	 * @param array<int, string>|null $format       Data format.
	 * @param array<int, string>|null $where_format Where format.
	 * @return int|false Rows affected or false.
	 */
	private static function update( string $table, array $data, array $where, ?array $format = null, ?array $where_format = null ) {
		global $wpdb;

		return call_user_func( array( $wpdb, 'update' ), $table, $data, $where, $format, $where_format );
	}

	/**
	 * Replace row.
	 *
	 * @param string             $table  Table name (with prefix).
	 * @param array              $data   Row data.
	 * @param array<int, string> $format Format array.
	 * @return int|false Replace result.
	 */
	private static function replace( string $table, array $data, array $format ) {
		global $wpdb;

		return call_user_func( array( $wpdb, 'replace' ), $table, $data, $format );
	}

	/**
	 * Get the latest settings row for a given connection owner email.
	 *
	 * @param string $mail Owner email.
	 * @return array|null Latest settings row.
	 */
	public static function latest_settings_for_owner( string $mail ): ?array {
		global $wpdb;

		$mail = sanitize_email( $mail );
		if ( '' === $mail ) {
			return null;
		}

		$table    = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$prepared = self::prepare_with_table_token(
			'SELECT id, user_mailid, app_id, access_token_ct, access_token_nonce, business_account_id, phone_number_id, phone_number, meta_webhook_subscribed
			 FROM {table}
			 WHERE user_mailid = %s
			 ORDER BY id DESC
			 LIMIT 1',
			$table,
			$mail
		);

		return self::row( $prepared );
	}

	/**
	 * Get latest settings row where webhook is subscribed.
	 *
	 * @return array|null Latest settings row.
	 */
	public static function latest_settings_with_webhook(): ?array {
		global $wpdb;

		$table    = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$prepared = self::prepare_with_table_token(
			'SELECT id, user_mailid, app_id, access_token_ct, access_token_nonce, business_account_id, phone_number_id, phone_number, meta_webhook_subscribed
			 FROM {table}
			 WHERE meta_webhook_subscribed = %d
			 ORDER BY id DESC
			 LIMIT 1',
			$table,
			1
		);

		return self::row( $prepared );
	}

	/**
	 * Get the latest settings row (any owner).
	 *
	 * @return array|null Latest settings row.
	 */
	public static function latest_settings_any(): ?array {
		global $wpdb;

		$table = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$sql   = self::prepare_with_table_token(
			'SELECT id, user_mailid, app_id, access_token_ct, access_token_nonce, business_account_id, phone_number_id, phone_number, meta_webhook_subscribed
			 FROM {table}
			 ORDER BY id DESC
			 LIMIT 1',
			$table
		);

		return self::row( $sql );
	}

	/**
	 * For each owner_mail, return only their latest settings row.
	 *
	 * @return array<int, array> Rows.
	 */
	public static function latest_rows_per_owner(): array {
		global $wpdb;

		$table = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$sql   = self::prepare_with_table_token(
			'SELECT t.*
			 FROM (
				SELECT user_mailid, MAX(id) AS max_id
				FROM {table}
				GROUP BY user_mailid
			 ) x
			 JOIN {table} t ON t.id = x.max_id
			 ORDER BY t.id DESC',
			$table
		);

		return self::results( $sql );
	}

	/**
	 * Get most recent OTP row for a given session and phone.
	 *
	 * @param string $session_id Session id.
	 * @param string $phone_e164 Phone in E.164.
	 * @return array|null OTP row.
	 */
	public static function otp_find_latest( string $session_id, string $phone_e164 ): ?array {
		global $wpdb;

		$table    = self::quote_table_name( $wpdb->prefix . 'nxtcc_auth_otp' );
		$prepared = self::prepare_with_table_token(
			'SELECT *
			 FROM {table}
			 WHERE session_id = %s AND phone_e164 = %s
			 ORDER BY id DESC
			 LIMIT 1',
			$table,
			$session_id,
			$phone_e164
		);

		return self::row( $prepared );
	}

	/**
	 * Get id of active OTP row for a given session and phone, if any.
	 *
	 * @param string $session_id Session id.
	 * @param string $phone_e164 Phone in E.164.
	 * @return int|null OTP id.
	 */
	public static function otp_find_active_id( string $session_id, string $phone_e164 ): ?int {
		global $wpdb;

		$table    = self::quote_table_name( $wpdb->prefix . 'nxtcc_auth_otp' );
		$prepared = self::prepare_with_table_token(
			'SELECT id
			 FROM {table}
			 WHERE session_id = %s AND phone_e164 = %s AND status = %s
			 ORDER BY id DESC
			 LIMIT 1',
			$table,
			$session_id,
			$phone_e164,
			'active'
		);

		$id = self::var_value( $prepared );

		return $id ? (int) $id : null;
	}

	/**
	 * Update OTP row by id.
	 *
	 * @param int   $id   OTP row id.
	 * @param array $data Data to update.
	 * @return void
	 */
	public static function otp_update_by_id( int $id, array $data ): void {
		global $wpdb;

		self::update( $wpdb->prefix . 'nxtcc_auth_otp', $data, array( 'id' => $id ) );
	}

	/**
	 * Insert new OTP row and return its id.
	 *
	 * @param array $data Row data.
	 * @return int Inserted id.
	 */
	public static function otp_insert( array $data ): int {
		global $wpdb;

		self::insert(
			$wpdb->prefix . 'nxtcc_auth_otp',
			$data,
			array(
				'%s', // session_id.
				'%s', // phone_e164.
				'%d', // user_id.
				'%s', // code_hash.
				'%s', // salt.
				'%s', // expires_at.
				'%d', // attempts.
				'%d', // max_attempts.
				'%s', // status.
				'%s', // created_at.
			)
		);

		return (int) $wpdb->insert_id;
	}

	/**
	 * Find phone binding row (if number already linked to a user).
	 *
	 * Returns an object because callers use object properties.
	 *
	 * @param string $phone_e164 Phone in E.164.
	 * @return object|null Binding row.
	 */
	public static function binding_find_by_phone( string $phone_e164 ) {
		global $wpdb;

		$phone_e164 = sanitize_text_field( $phone_e164 );
		$table      = self::quote_table_name( $wpdb->prefix . 'nxtcc_auth_bindings' );
		$prepared   = self::prepare_with_table_token(
			'SELECT * FROM {table}
			 WHERE phone_e164 = %s
			 LIMIT 1',
			$table,
			$phone_e164
		);

		return call_user_func( array( $wpdb, 'get_row' ), $prepared );
	}

	/**
	 * Mark an existing binding as verified if it has no verified_at yet.
	 *
	 * @param int $id Binding id.
	 * @return void
	 */
	public static function binding_mark_verified_if_empty( int $id ): void {
		global $wpdb;

		self::update(
			$wpdb->prefix . 'nxtcc_auth_bindings',
			array(
				'verified_at' => current_time( 'mysql', 1 ),
				'updated_at'  => current_time( 'mysql', 1 ),
			),
			array( 'id' => $id )
		);
	}

	/**
	 * Upsert a binding mapping a user to a phone number.
	 *
	 * @param int    $user_id    WP user id.
	 * @param string $phone_e164 Phone in E.164.
	 * @return void
	 */
	public static function binding_replace( int $user_id, string $phone_e164 ): void {
		global $wpdb;

		self::replace(
			$wpdb->prefix . 'nxtcc_auth_bindings',
			array(
				'user_id'     => $user_id,
				'phone_e164'  => $phone_e164,
				'verified_at' => current_time( 'mysql', 1 ),
				'created_at'  => current_time( 'mysql', 1 ),
				'updated_at'  => current_time( 'mysql', 1 ),
			),
			array( '%d', '%s', '%s', '%s', '%s' )
		);
	}

	/**
	 * Insert a message history row for outbound template sends.
	 *
	 * @param array $data Row data.
	 * @return void
	 */
	public static function history_insert( array $data ): void {
		global $wpdb;

		self::insert(
			$wpdb->prefix . 'nxtcc_message_history',
			$data,
			array(
				'%s', // user_mailid.
				'%s', // business_account_id.
				'%s', // phone_number_id.
				'%s', // template_name.
				'%s', // template_type.
				'%s', // template_data.
				'%s', // status.
				'%s', // status_timestamps.
				'%s', // created_at.
				'%s', // sent_at.
				'%s', // meta_message_id.
			)
		);
	}
}
